// Copyright (c) 2022 Advanced Micro Devices, Inc. All rights reserved.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

#include <cstring>
#include <fstream>
#include <iomanip>
#include <iostream>
#include <string>

/// Number of bits in the state space of MT19937.
constexpr unsigned int mexp = 19937;
/// Number of bits in an unsigned int.
constexpr unsigned int w_size = 32;
/// Minimum number of unsigned ints to represent mexp bits.
constexpr unsigned int p_size = (mexp + w_size - 1) / w_size;

/// String in this array contains the constants to skip <tt>2 ^ 1000</tt> states.
/// Values are produced using minipoly_mt19937.c as distributed in:
/// http://www.math.sci.hiroshima-u.ac.jp/~m-mat/MT/JUMP/jump_ahead_1.02.tar.gz
/// Size of tables is <tt>mexp + 1</tt> to account for null-terminator.
namespace
{
extern const char jump[mexp + 1];
} // namespace

/// Set the coefficients of polynomial \p pf based on <tt>p</tt>.
void set_coef(const char p[mexp + 1], unsigned int pf[p_size])
{
    for(size_t i = 0; i < mexp; i++)
    {
        size_t j = mexp - i - 1;
        if(p[i] == '1')
        {
            // Set coefficient j of the polynomial pf.
            constexpr unsigned int log_w_size  = 5U;
            constexpr unsigned int w_size_mask = (1U << log_w_size) - 1;
            pf[j >> log_w_size] ^= 1U << (j & w_size_mask);
        }
    }
}

int main(int argc, char const* argv[])
{
    if(argc != 2 || std::string(argv[1]) == "--help")
    {
        std::cout << "Usage:" << std::endl;
        std::cout << "  ./mt19937_precomputed_generator "
                     "../../library/include/rocrand/rocrand_mt19937_precomputed.h"
                  << std::endl;
        return argc != 2 ? -1 : 0;
    }

    const std::string file_path(argv[1]);
    std::ofstream     fout(file_path, std::ios_base::out | std::ios_base::trunc);
    fout << R"(// Copyright (c) 2022 Advanced Micro Devices, Inc. All rights reserved.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

#ifndef MT19937_PRECOMPUTED_HPP_
#define MT19937_PRECOMPUTED_HPP_

// Auto-generated file. Do not edit!
// Generated by tools/mt19937_precomputed_generator

)";

    fout << "constexpr unsigned int mt19937_mexp   = " << mexp << ";\n";
    fout << "constexpr unsigned int mt19937_p_size = " << p_size << ";\n";

    unsigned int pf[p_size] = {};

    fout << "\n"
            "// clang-format off\n"
            "static constexpr unsigned int mt19937_jump[mt19937_p_size] = {\n    ";

    set_coef(jump, pf);

    for(size_t i = 0; i < p_size; i++)
    {
        fout << std::setw(10);
        fout << pf[i] << "U";
        if(i + 1 < p_size)
        {
            fout << ", ";
            if((i + 1) % 8 == 0)
            {
                fout << "\n    ";
            }
        }
    }

    fout << "\n};\n"
            "// clang-format on\n";

    fout << R"(
#endif // MT19937_PRECOMPUTED_HPP_
)";
}

namespace
{
const char jump[mexp + 1] = "1001011000011110011001111001101011110100111110110101110000101001111100"
                            "110000000000100110111111"
                            "0001001110101111000000001000100101001101011100101100011111101001111100"
                            "110010101100101001100100"
                            "1110101001000111011001111111000011001001110000110101100101101010011000"
                            "101110001111101100111111"
                            "1101001010111011110101011010100100110001001100100010100010000011110001"
                            "001101011011110000101011"
                            "1011010001110000100011110001001010000110010000011010101011101000110000"
                            "010111110101100010000110"
                            "1111000011001111111100111110001111101000000111010100111001011011011101"
                            "100111001100011011000010"
                            "1011101101001101111010111001000100100010101101100100100100111101100100"
                            "100100111000010101011011"
                            "1011111011100110100111100011000110011110011101010000110001001101110011"
                            "010110101000010111100110"
                            "0000101100111100001011011001011011001010110110010000001101011111110000"
                            "110101000101111000101001"
                            "1111101000111001110010000111000001110001100001101000111011010100111011"
                            "011110000010101000110010"
                            "0110001111101001001110100110111101010111000011101010110111001000101011"
                            "101001011100010011001010"
                            "1111110110000101110100100010000111100100001101101111100000010111111001"
                            "010000110100000110110011"
                            "1101001100101011000101000010110110100110010000110111110110100100101101"
                            "010011110100100111000101"
                            "1100100011001111000000000100010100111100100001000010110111011110110011"
                            "011001001001001010110110"
                            "0100111010110001000101101001100001011100100010100111101101100010000000"
                            "110011100101011010101101"
                            "1000010101000001111011110100011110010001001101100000001000000100110011"
                            "011110001111000010011001"
                            "0000001010011110000110111011100111000001110011000101010110010001110111"
                            "100100111100011111101111"
                            "0100000101110111001011010110101001010111101010110000000101011001010101"
                            "010011100101000011000011"
                            "0000001111101110111101000101011010000110110000000100111010111011011100"
                            "110100100100011101001110"
                            "1111110101001011001010101010011001100101010000100101011100010111101001"
                            "001011011110000001100110"
                            "0111000000110000011101101101100100000100100000101001001011110000001011"
                            "000010000110101101110010"
                            "1101101100000101110010101011101000101001101101000011010000100101001101"
                            "111100010011001000001011"
                            "1101011100010101001101111010101101111110100111000010101101010110010001"
                            "100110110010111100110000"
                            "1000111011100101101101001111011001101100011000001001000111011000111101"
                            "111100010010100101101000"
                            "1000000101010100001100111011011111001001011001111110110111011010010000"
                            "000101111110101110001100"
                            "1110000010110000011000111100000111110011100010010001101100010100110010"
                            "011001001100010010100010"
                            "0011000111001011010011011000110111010101011110011111110100100001100011"
                            "111001100010101010010000"
                            "1100010011101011101101000010010010110001100110001110000111110001010001"
                            "010100010110110110011110"
                            "1100100001011000101111101001011101100001011011110011111001110111101001"
                            "001001111010001000101010"
                            "0011000001011110100101011000111110101111101000111001000011100011011010"
                            "101011101001000101101011"
                            "0110011010100111110111110101110111101001000110101011101000011011011110"
                            "100110011001000101011010"
                            "1101011100000100000111101010001011100010100010001010100111001101011010"
                            "111101100000000001000110"
                            "0000000111001110011010011111000011001000100001001111011011010100111001"
                            "110011111100101101000101"
                            "1110101101011100101011001001111011010011101001100011101110001000001001"
                            "100011100011010010011110"
                            "1110110100011001101000010000101100111001110011010111000001011000101111"
                            "100101100100111101010011"
                            "1010010011101010001011011010011111100000110101111100000010011001111111"
                            "110101001110111110010110"
                            "0010101101101010000111101011110100011001011110001110110010000101101101"
                            "001011100100010101001001"
                            "0110100010000001011000100011110010000001001100000001000011101111000110"
                            "011010011101101111010100"
                            "1101010110011010011101010111001001100100010011111100010101001010100100"
                            "001100011111001010000000"
                            "0110101100010100100001000111011000100001011001001111011100110110001101"
                            "100010100011011001010101"
                            "0010001001001001101011110101110010111100100101110001001000110010010000"
                            "100110100110000001100010"
                            "0111010001000101101111000100001010000000110000000010110010101000101110"
                            "100111010100100000101010"
                            "0001101000100011100000011100101010011101111100010111000101110110010101"
                            "111011100110110010001010"
                            "1110011111000000110101010100000000111101010110110010111100001100001011"
                            "101101111110110010010101"
                            "1111011000101111101101111010000011110110000000111000001110001001010111"
                            "101001110100101000010111"
                            "0000000100110000101011011100110010110110011110111001101100111010010001"
                            "000111111100001010000000"
                            "1101100010010110100111011111101111011001000000110110110101001011101100"
                            "111101101000100011010000"
                            "0101111001001010110101011000100000111101010101000011001111000010000111"
                            "100011000100011110000111"
                            "0110111100001000000011100101101101100100110001101001011000100110110010"
                            "100111000101001110111110"
                            "0110011100000010111100010010110111001001010001101001110000010010011101"
                            "101110110010011111000010"
                            "1000111011110110000101100110011101010011010001001101001111100000100010"
                            "000111111001011110000000"
                            "1000100010001101111101110001011010111100001001000110101100101011001001"
                            "111001000101101101101000"
                            "0010101101111101111100000100011011110110101101111110111111000100000101"
                            "101110010110001101100001"
                            "0011001000011010000100101010100011011011011110001001100110011110101011"
                            "010110001110000101110000"
                            "1111101001100110000101000101100001100101010110100100111111011101110111"
                            "111110000101010001001100"
                            "1011011010001110011110000001011000101100011011111000001000110111001000"
                            "100111010010111011000111"
                            "1110110010011111110110111110101101111101101000011000110001100000110000"
                            "011000111111001110111001"
                            "0111001010010110000011110001000111001101011011101001110110100000011010"
                            "000000100001011011000101"
                            "0000011100110010111000100001110100100111111110100010010001101000111011"
                            "011000000011111011110011"
                            "0001011011101100100011011110000111011101011111000111101111000100101100"
                            "000100011010100011001010"
                            "1000001111111000100100001111000011001100100110110110011101001011111101"
                            "011100110100110000010101"
                            "0100000100101011001111001010011101000100100011001101110100110010111000"
                            "111010011110111100001100"
                            "1001011000101110110101011011100011000110110110011110011001110110101000"
                            "101010110001000111111110"
                            "0000001001100011110011101010111111110111010110101101110101101001000111"
                            "100010101010100101111111"
                            "0010110110100100101100100100101001010110110000000111011110001101111001"
                            "000110111101001100011110"
                            "0000100101000011111110100001010010011110110011101010100011101100011100"
                            "011010010001010101100101"
                            "0011010100000110010001101111110111100110111100110011011010000101011100"
                            "011111100001000001000001"
                            "0110000010001111010010000010010001000101000010001000100111100011001110"
                            "001100000101111101111101"
                            "0100100110000010000100100000111100111100001010101100011011100100101001"
                            "100011110101100110101001"
                            "1110010011111001110011011011100011010100001111011101100011010110110011"
                            "000000101101101001011011"
                            "1000111001101100000010011011000110101010110001100100011011111100101001"
                            "111100111011000100100101"
                            "0001010010111000101101111000001100000101101000010110010100000001001110"
                            "011110110111001110011001"
                            "0100100111111010001001011000111001110001011001011011000110101001110111"
                            "010011110101001011111110"
                            "0000101111100010011101101101110010100110110100011001101101100000010011"
                            "010110111001011101011100"
                            "0111001110010111011010111100001111101101011111001101011001001011011111"
                            "101101100101011010001010"
                            "0100100100010011000011101110010100100100001000001110101111101000111101"
                            "111000111010011001010111"
                            "1000100110101001100010000101101010100110110011111011000010010111010110"
                            "110101011000101111010011"
                            "1111101010001101010000010000000000011000110111010011100010110001000011"
                            "010100000011110110011011"
                            "1100110100000011110100000010011000111111000010110100010100001100100010"
                            "010101011001010101001101"
                            "1011001110110001110101001100110000001110101011001100100010111000101011"
                            "110111110000000000100011"
                            "1010001011011011111011000100111000101110101011100001100000110010100111"
                            "111110111011100110001101"
                            "1111011110000001001110101000110111111101010111111010101101101111110110"
                            "011111010101011110110001"
                            "1010001111101110110110101111011111111000110111001100101101010111000000"
                            "001110001100100010011101"
                            "1010101011010110101100001100100110000001011011010100010001001111010111"
                            "010110100111011111101110"
                            "1100111110110010100101110111110101001110110001101100011010010111011110"
                            "000001100010100010000100"
                            "1100000001000010111001001110000100100100011110011100000101000011010101"
                            "101110110100011011000101"
                            "0000111010010000101100001011001011110111001000111011101011000011101111"
                            "100011100100011100100100"
                            "1111101010110010000000110001010111110100110100111111100111011101101011"
                            "100001011010110111001101"
                            "1101101010010010110000110001011010110100001110101101100100011111000101"
                            "101010111101101000000100"
                            "0000111101010101110100111011100001110100000001001101101110111001011100"
                            "100110111101010101011010"
                            "0100111000011100101000010010111111000000011111100000001100110101010001"
                            "101000001110010000001011"
                            "1111001101011011111110110001011011101101011111111110010001011110101000"
                            "110010011011001111011100"
                            "0100011101000000000110110011011111000111011110000001010011000001010111"
                            "011110111010000110000001"
                            "0011010100110101101010110100001000001001111111100110011101110110011110"
                            "100010100000001011001110"
                            "1100101000101111101000001111101011101001100001100101010001001101101001"
                            "111110111001010001000101"
                            "0111000010111111011001101100110110110111111100011101100101100111001000"
                            "100010001001010010000101"
                            "1101000110010111000011110101010111110110011100110101011000100111010111"
                            "010100010101110100110000"
                            "1111110001000100011101011110011010110011110000110100101010001000011111"
                            "110001110111010010010100"
                            "1101010001001101001011010010101001111100010100110011011100110100010011"
                            "111011110001101001101010"
                            "0110101100100011000000000000110000110100100010110100001011101001010011"
                            "111011011010011000110110"
                            "1100110101111010111100001101011001101110000010001110101111000000111111"
                            "010110111100111000110100"
                            "1110111100000001001001000000111100010111100001000111100100101001001010"
                            "001111100100101101000001"
                            "0001000011111111111011111101110100111100001101100110010011000111110001"
                            "001110110010011110000010"
                            "0100001110101001100101000100111111110010101101101001110010100101100011"
                            "000000101011111110100000"
                            "1000100110010010110000010001101000010011110011011011100010010110111001"
                            "110111110011001100011111"
                            "1110011101000110111101001111010100111110000100101001100001011000001001"
                            "110011110010100011110010"
                            "1011001110111100001010010001100101001001000100111010100100111100111011"
                            "101000100010100011110100"
                            "0010100001000001010100111000011010010010010100010010001010000111100101"
                            "001110010100100110011010"
                            "1001001101100010111010001000001101111000111101011001100100110011001100"
                            "000000110111101101011100"
                            "1010010011000111101011101011110111001011111001100011100111101100110100"
                            "101110001011101111010111"
                            "0101001100110010100101010000010100110100111111011111000111111101100100"
                            "011000000011001100011010"
                            "1001100110111010001010000111001110001001011100000011011010110001011011"
                            "111001101000011111000011"
                            "0010100011110101111010000010000011100111000110010010011101110010111011"
                            "010101001011001011001111"
                            "0100101011000000111010111010001100100111000001111010111111100111001001"
                            "011011010011000001010010"
                            "1110101010110110111101000111001000100111111110011011101111100011010010"
                            "000111101000000010000001"
                            "1101101110010000110000001100010100111000101100011110111110010110111111"
                            "000111110000110000010101"
                            "1010001000001000101000010000011000111001001011100000101111001111000001"
                            "111111011011001110100001"
                            "1011101110101010000001111000100000000010100010000000011100111111000010"
                            "100000111110010001111110"
                            "1101010101101010100110101101011001011101010011101110101100011101011011"
                            "100101000010110101101110"
                            "0101111111101000000001100100011111000101100010011011011010110110010001"
                            "010011011010111110010011"
                            "1010010001110101111010010100010010000000111001001000010111010000010101"
                            "000010010100011010001110"
                            "1100001011011010110100111011010011100001000011010111001101011100111001"
                            "101010000010100110110101"
                            "1010010010100111110111011100110111010110111011100100000100000011001011"
                            "001110110011111010110010"
                            "0000110010001010010000101101001110011101110011100010000111110010101001"
                            "100011110011110000101110"
                            "1111100111111010111101010000100100110011100010011000101110100101011111"
                            "000001000001011110101000"
                            "0100101110111010001001011001011100000100000110011110000100111100101010"
                            "000111011011111000000000"
                            "1100100000101110000001001111011000110011111110111111000010101010111001"
                            "111100101100110001110110"
                            "1010111000111010011000010100001110100000001001000010101101100111110100"
                            "110011110101111010001110"
                            "0010011000000000101100111011011010100010011110110111001001111100101110"
                            "001100010011111010101101"
                            "0011110111101000011010000100101111000101100001010101000011001100111110"
                            "000010011010110011110010"
                            "0000010001011011011001111111001101101101010100100101001010110100001011"
                            "100100000110110001101011"
                            "0011011111111111101100111101110011010010011111010101001001111010010011"
                            "110100011100101001000101"
                            "1010011010010111111100000111101000010100000010010100011001011101010001"
                            "011011110101000010100010"
                            "0000101111110101011111101100101101110101000111101010000011100101001111"
                            "100000011101110111100011"
                            "0011010000100000101110111000101001010000101001011001000101000110101000"
                            "110101101111100111010110"
                            "0100011101101100101011111010110110001100010100000110110110110100001100"
                            "100000001000001101001000"
                            "0111100100100110001100000111010100010111101001001111100110111100111111"
                            "001110100010100101110101"
                            "1110110011110010111010111101111000101101001010100100111010101001001111"
                            "100010111111010010011001"
                            "0101101001001101010010011110111000111100000000111111010010000111010001"
                            "010000111111001100111101"
                            "1011000011110100011100001101010000000011000111101111100011101101100000"
                            "000001101000111110111010"
                            "1011000001111100101010011010111110001011110001110000111001011001001000"
                            "000010000001110101000111"
                            "1111010001001111110111110011001011001000001100101010110010111111011111"
                            "100111000110101101010110"
                            "0001001011110101011101111000111000011101000110011010010111010011111110"
                            "001000111011100000000001"
                            "0110100110010100111010111100011010110010100110100111010001010001111010"
                            "011110011011100000001000"
                            "1000111100110111000010110000011010100101101111101011011000110100011001"
                            "111000001001001100011111"
                            "1011000011001001110101001101100101100010110101111001111110111010101001"
                            "001000001110001000100010"
                            "0100111000011001111010100101000000011110110110000011000101011110111000"
                            "101010110011010110100100"
                            "1010011001110111110010100001001100110110011011101001000101110100001100"
                            "000000011000100010000011"
                            "1100111010100100001101101110101011001101110001001111011010001011000001"
                            "011011010100001100010111"
                            "0000000100010010101111111010010110010100111100010111011111101011001110"
                            "110110101100011100011011"
                            "0110101001011111001011010011001111100111110101000000011111111011000101"
                            "010111000110110000001000"
                            "1011001011011011100001011100101000110110001100010011100000111000000011"
                            "101001110000010110111011"
                            "1000011101110111101010000111101011110001001001000011110010111111000011"
                            "101001010000101000110000"
                            "1111100100011100111111100010100100000100101001111011000100010001011000"
                            "110100101010010110010101"
                            "1101111100011010010100011110101001010011010001100001100001011100101010"
                            "111011110100001100101111"
                            "1110111011001111101000000001011010101100011111010011101011010001000011"
                            "000000011101110010111010"
                            "1111010010000111010010001100000101010111101010111001010010001010011010"
                            "000100001011000000101100"
                            "1111101001011100001101101000011000111100110000101011011001100110111010"
                            "010010100011101011101010"
                            "0110001111101011010111111100001011010000010010101001010010000110101000"
                            "001011110000010110110010"
                            "1101000001101000111110101001111010001010110111111011011101011011001011"
                            "111111110110011000000100"
                            "0000000110100011111001011101101100010110110110000000000010001010100110"
                            "001011011011001011111100"
                            "0011110100110010101001010110111101010010101101110000101100101100100011"
                            "001011000101100001000010"
                            "0110111011101001010110000001100100110111011101110101000000100100110100"
                            "100101100000001000000011"
                            "1110100110001001110111101100110110101011101101111000100001111101011100"
                            "001110010011000001110110"
                            "1100011110110010110011101101100111010000010100001010010011001001100101"
                            "101101100010000101011100"
                            "1110101111100011100011001000010011000111011010101100011110101111010110"
                            "001010001011111001111010"
                            "1011111010101011011000110111101001111001011111110010011111011000000100"
                            "011110001010000101011101"
                            "1000010010101000110011101100111011110100011101000101000110111110110111"
                            "010001001100011100110011"
                            "0011010000000100011010010100011110001000100001011101101101011111001000"
                            "100110000000010111100011"
                            "1011000001100000101110110100101100010001001101100010011100111011101100"
                            "011010001011111101101001"
                            "1100100010111101101000010111100110111000000100100111011010110001111001"
                            "101001011011001100111000"
                            "0010101000110011101010010001101010100010101001000001001100111011111000"
                            "000101110111101010100101"
                            "1000011110101000000010011111100110110001110001001001000111101110101101"
                            "000111110010011110011000"
                            "0011100111010110000011010001011111110010101011110111000000010101010001"
                            "001001011111011001001110"
                            "1101010100101001101111001100001010101110101111001100001110101111110010"
                            "011111110010111001101000"
                            "1101000101001110101011101100010110011111101001001110101001011000101101"
                            "110100011011101000111100"
                            "0011011100010011100100001001001110100001110100110111110010101001001110"
                            "011111011110001111001000"
                            "1000100110100100001001100111011000001000010001101001011010001100111001"
                            "001010100111111100001010"
                            "1011000010100000111001011011011010011100110100001100001111110110111100"
                            "011000000111011001111000"
                            "1011111000010100111010110101110101101011010100100000101001110011111110"
                            "011111111001000011111001"
                            "1010001111100110110101011000100010101001100000010000010111100001101001"
                            "001011100101101100001101"
                            "1111100101110101110100100001010000011001100001100011110011010111100001"
                            "110111111001111001111111"
                            "1110111100101001000000010010100011101100011111101011111101111111110111"
                            "000100110011000100111001"
                            "0000011111101110100101011100001010101111000001100110110001110001000100"
                            "100100100100101000000111"
                            "0110000010111000110000110101000001110111101100000110111110111011100000"
                            "011001001100101000111101"
                            "0100011110000110110110001010011101010110011101100011011001010110101100"
                            "000111001111000110111001"
                            "1101001011010111101111011011100010111110101110001101011110101111000101"
                            "011111010011111000101110"
                            "0100110000100000101001010110110001100100111110110000101001001101001100"
                            "000011010010001100111110"
                            "0011010001000000100011000111010100010000010010011000010100110101111010"
                            "100001010000000101001001"
                            "1110111110001010111011101111000001001111011100010010011110101011011100"
                            "001101101100011010111100"
                            "1110010010111111010011101010110111001100111010110011110011011011111000"
                            "000011100111100100101111"
                            "0100010010111011101100111111101101110000000001110100111110001100111111"
                            "010111011001001111010100"
                            "1000001100110101100110100001011101101100010111111111101111101000001001"
                            "011010111010000001001001"
                            "1011001001001100011111000101110000011001110100110011101000010110001010"
                            "011001100010111101001011"
                            "1110011111000111111011101011111001010100000111001010010000000001000111"
                            "001100111111101000101100"
                            "0111100000100110011011111110111100000000011001001011110111001011011001"
                            "100110011001101000010001"
                            "0000101110011111111111011011000001111010111010010000101010001000001110"
                            "001110110111101101100110"
                            "0010101100011011001111101110000110111100010010010000101010001000110111"
                            "110001000100010011111101"
                            "1101001010001011101101101111001110010100001111101011001000111100111010"
                            "110010000111111111001001"
                            "0001001101011111001000111101101001011111100101100100001100001101110000"
                            "100111101000000010001110"
                            "0110000011011101001111010101101101011011101111010110000011010010010001"
                            "110000010101100000111000"
                            "0010001011100011111001000101111111110000010000000111001101100111000100"
                            "110001100000111011110000"
                            "0001111000011111100111001111000100010101010101101000100110101110110100"
                            "100101101010000110001001"
                            "0000100110001001000111100111101111110110110110000100000101011010110000"
                            "110000101010010101001000"
                            "0010110111110101101001110001101011011011011101101111001111010100001101"
                            "010001111111100111000111"
                            "0000011111001001101111111010011000111101010001110011001001110000001100"
                            "100110010000010111001000"
                            "0101101000010101011010111100011001100100000110000111001001100010110011"
                            "001001001101001001101101"
                            "1000100110001010010110111110001111111100110101101111111100101101111101"
                            "111000010011000110100010"
                            "0111100010011001010111100000000110011001110101110111001111100110100100"
                            "100010000011111110010111"
                            "1101011000111011100010111000011010010111101100111010101001010001101101"
                            "110000011110011000111100"
                            "0001010011011001101111000110000110010110001011001000001100000101100000"
                            "110010110110101010001110"
                            "1111000010100111011001001001011101000010001111100011111000011101100100"
                            "000111101010110001001100"
                            "111010101";
} // namespace
